
/*
 * Frequency Meter with Non-Contact Mains Frequency Indicator
 * Compiler: AVR GNU C Compiler (GCC)
 * Written in AVR Studio 6
 * Micro: ATmega8A
 * Created: Feb 20, 2021
 *  Author: Mahmood Alimohammadi
 * Description: Measures and displays the  frequency(up to 6MHz) 
 *of the external source connected to PIN 6(PD4) or the 
 *Non-Contact AC mains Frequency in the Non-Contact Mains Frequency mode. 
 *Timer1 serves as a time reference and Timer0 is employed as a
 *counter. Timer2  is used to toggle PIND0 to produce sound.
 */
//Micro ATmega8A 
#ifndef	F_CPU	16000000UL
#define F_CPU	16000000UL
#endif
#include <stdio.h>
#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <stdlib.h>
#include <avr/interrupt.h>

#include "lcd.h"
/*----------------------------------------------------*/
//External clock source on T0 pin. Clock on rising edge
#define COUNTER0_ON TCCR0 |= (1<<CS00|1<<CS01| 1<<CS02)
#define COUNTER0_OFF  TCCR0 &= 0B11111000
#define COUNTER0_CLEAR TCNT0 = 0
// Start Timer 1 at Fcpu /256
// Configure Timer 1 for CTC mode//
#define TIMER1_ON TCCR1B |= (1<<CS12|1<<WGM12)
#define TIMER1_OFF  TCCR1B &= 0B11111000
// Start timer2 at Fcpu /32 for toggling PIND0
#define SOUND_ON TCCR2 |= (1<<CS21)|(1<<CS20)
#define SOUND_OFF  TCCR2 &= ~(1<<(CS21)|(1<<CS20))

#define LED_ON PORTD |= (1<< 1)
#define LED_OFF PORTD &= ~(1<< 1)
#define Sensitivity_Low bit_is_clear(PIND,PD2)
#define Sensitivity_High bit_is_set(PIND,PD2)
#define Mains_Freq bit_is_clear(PIND,PD3)

long freq, count;
/*----------------------------------------------------*/ 
void display_freq(void);
 
/*----------------------------------------------------*/ 
int main(void)
{	// Set PIND0 and PIND1 as output for LED1 and sounder
	DDRD |= (1<< 0)|(1<< 1);
	//Set PIND2 as input for Sensitivity and PIND3 as input for Mains_Freq 
	DDRD &= ~((1<< 2)|(1<< 3));
	//activate pullups for PIND2 and PIND3 
	PORTD |= ((1 << 2)|(1<< 3)); 
	
	LCD_init();
	COUNTER0_ON;
	TIMER1_ON;
	// 16000000/256=62500
	// Set CTC compare value to 1 Hz at 16 MHz AVR clock , with a prescaler of 256
	OCR1A=62500-1;
	//Enable overflow interrupts of Timer0, Timer1 and Timer2
	TIMSK = 1<<OCIE1A|1<<TOIE0|1<<TOIE2;
	//set global interrupts enable
	sei();
 /*----------------------------------------------------*/
	while(1)
    {
		display_freq();	
		if (freq>5 && freq<52  && Mains_Freq && Sensitivity_High)
			{
				LED_ON;
				SOUND_ON;
				_delay_ms(200);
				LED_OFF;
				SOUND_OFF;
			}
		
		if (freq>45 && freq<52 && Mains_Freq && Sensitivity_Low)
			{
				LED_ON;
				SOUND_ON;
				_delay_ms(200);
				LED_OFF;
				SOUND_OFF;
			}
		}		
	
}
/*----------------------------------------------------*/
void display_freq(void)
{	
	LCD_Clear();
	if(Mains_Freq)
		{
			LCD_goto(1,1);
			LCD_Text("Mains Freq.");
		}
	else
		{
		
			LCD_goto(1,1);
			LCD_Text("Frequency Meter");
		}
		
	LCD_goto(2,1);
	LCD_Text("F: ");
		
	LCD_Long(freq);
	LCD_Text("Hz");
	
	_delay_ms(200);
}
/*----------------------------------------------------*/
/* Interrupt Service Routine of Timer1 as timer*/
ISR(TIMER1_COMPA_vect)
{	
	COUNTER0_OFF;
	TIMER1_OFF;
	freq=count*256;
	freq=freq+TCNT0;
	if(Mains_Freq)
	freq=freq*2;
	COUNTER0_CLEAR;
	count=0;
	COUNTER0_ON;
	TIMER1_ON;
}
/*----------------------------------------------------*/
/* Interrupt Service Routine of Timer0 as counter*/
ISR(TIMER0_OVF_vect)
{
	count++;
}
/* Interrupt Service Routine of Timer2 */ 
ISR(TIMER2_OVF_vect)
{
	PORTD ^= (1 << 0); // Toggle PIND0 for sound generation
}
//----------------------------------------------------
//End of Program